#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "../dsplib/dsplib.h"
#include "../nr_lib/nr_lib.h"
#include "../Ninereeds Broadcast Lib/Ninereeds Broadcast Lib.h"

#pragma optimize ("a", on)

#define MAX_TRACKS	8

float const oolog2 = 1.0 / log(2);

NR_LIB_WAVEFORM_PAR	   (paraWForm,		"Waveform",		"Shape of Waveform"									)
NR_LIB_ATTACK_PAR	   (paraAttack,		"Attack",		"Attack time in ms", 500									)
NR_LIB_ATTACK_PAR	   (paraDecay,		"Decay",		"Decay half life in ms", 500								)
NR_LIB_AMPLITUDE16_PAR (paraSustain,	"Sustain",		"Sustain Level (0=0%, 8000=100%, FFFE=~200%)", 0x8000		)
NR_LIB_ATTACK_PAR	   (paraRelease,	"Release",		"Release half life in ms", 500							)
NR_LIB_ATTACK_PAR	   (paraBend,		"Bend",			"Pitch bend half life in ms", 500						)
NR_LIB_FRAC_EFFECT_PAR (paraEffectLow,	"low effect",	"Fractal Effect (Low)"								)
NR_LIB_FRAC_EFFECT_PAR (paraEffectHigh,	"high effect",	"Fractal Effect (High)"								)
NR_LIB_FRAC_DEPTH_PAR  (paraDepth,		"depth",		"Fractal Depth"										)
NR_LIB_NOTE_PAR		   (paraNoteInit,	"NoteInit",		"Note Initialise"									)
NR_LIB_NOTE_PAR		   (paraNote,		"Note",			"Note"												)
NR_LIB_AMPLITUDE_PAR   (paraVolume,		"Volume",		"Volume (0=0%, 80=100%, FE=~200%)", 0x80	)


CMachineParameter const *pParameters[] =
{
	// global
	&paraWForm,
	&paraAttack,
	&paraDecay,
	&paraSustain,
	&paraRelease,
	&paraBend,
	&paraEffectLow,
	&paraEffectHigh,
	&paraDepth,
	// track
	&paraNoteInit,
	&paraNote,
	&paraVolume
};

#pragma pack(1)

class gvals
{
public:
	byte wform;
	word attack;
	word decay;
	word sustain;
	word release;
	word bend;
	word effect_low;
	word effect_high;
	byte depth;
};

class tvals
{
public:
	byte noteinit;
	byte note;
	byte volume;
};

#pragma pack()

CMachineInfo const MacInfo =
{
	MT_GENERATOR, MI_VERSION, 0,				// type, version, flags
	1, MAX_TRACKS,								// min, max tracks
	9, 3, pParameters,							// num globalpars, num trackpars, *pars
	0,    NULL,									// num attribs, *attribs
	"Ninereeds NRS02", "NRS02", "Steve Horne",	// name, short name, author
	"Edit Ninereeds NRS02"										// command menu
};

class mi;

class CTrack
{
public:
	void Tick(tvals const &tv);
	void Stop();
	void Reset();
	bool Generate(float *psamples, int numsamples);

public:

	c_ADSR_Track   *f_Level;
	c_Pitch_Track  *f_Pitch;
	c_Frq_To_Phase *f_Frq_To_Phase;
	c_Waveform     *f_Waveform;  //  This just duplicates the pointer from mi::f_Waveform
	c_Fractal      *f_Fractal;   //  This just duplicates the pointer from mi::f_Fractal

	float f_Buffer [MAX_BUFFER_LENGTH];

	CTrack (c_ADSR_Global *p_Level_Global, c_Pitch_Global *p_Pitch_Global )
	{
		f_Level        = new c_ADSR_Track (*p_Level_Global);
		f_Pitch        = new c_Pitch_Track (*p_Pitch_Global);
		f_Frq_To_Phase = new c_Frq_To_Phase;
	}

	~CTrack (void)
	{
		delete f_Level;
		delete f_Pitch;
		delete f_Frq_To_Phase;
	}

	mi *pmi;
};

class mi : public CMachineInterface, public c_Broadcast_Listener_CB
{
public:
	mi ();
	virtual ~mi();

	virtual void Init(CMachineDataInput * const pi);
	virtual void Tick();
	virtual bool Work(float *psamples, int numsamples, int const mode);
	virtual void SetNumTracks(int const n);
	virtual void Stop();

  virtual void Command (int const i);
  virtual void Save    (CMachineDataOutput * const po);

  virtual void Transpose_Notes  (int p_Semitones);

  virtual void All_Notes_Off    (void);
  virtual void All_Notes_Ignore (void);

  virtual void Set_Control      (int p_ID, int p_Value);

  virtual void Tick2 (void);
  virtual void Tick2 (int p_Channel, c_Broadcast_Params *p_Command);
  virtual bool Work2 (float *psamples, int numsamples, int const mode, int offset);

	//  Do the following all need to be pointers?
	
	c_ADSR_Global  *f_Level;
	c_Pitch_Global *f_Pitch;
	c_Waveform     *f_Waveform;
	c_Fractal      *f_Fractal;

  bool  f_Listen_Enabled;
  int   f_Listen_Channel;

	int numTracks;
	CTrack *Tracks[MAX_TRACKS];

	tvals tval[MAX_TRACKS];
	gvals gval;

};

DLL_EXPORTS

NR_STD_GENERATOR_SET_NUM_TRACKS
NR_STD_GENERATOR_STOP

mi::mi()
{
	f_Level    = NULL;
	f_Pitch    = NULL;
	f_Waveform = NULL;
	f_Fractal  = NULL;

	for (int i = 0; i < MAX_TRACKS; i++)
	{
		Tracks [i] = NULL;
	}

	GlobalVals = &gval;
	TrackVals  = tval;
	AttrVals   = NULL;

  f_Listen_Enabled = false;
  f_Listen_Channel = 0;
}

mi::~mi()
{
  if (f_Listen_Enabled)
  {
    NR_Stop_Listening (f_Listen_Channel, this);
  }

	for (int i = 0; i < MAX_TRACKS; i++)
	{
		delete Tracks [i];
	}

	delete f_Waveform;
	delete f_Pitch;
	delete f_Level;
	delete f_Fractal;
}

void mi::Init(CMachineDataInput * const pi)
{
	f_Level    = new c_ADSR_Global;
  f_Pitch    = new c_Pitch_Global (pMasterInfo->SamplesPerSec);
	f_Waveform = new c_Waveform;
	f_Fractal  = new c_Fractal;

	f_Level->Set_Samples_Per_Sec (pMasterInfo->SamplesPerSec);

  for (int i = 0; i < MAX_TRACKS; i++)
	{
		Tracks [i] = new CTrack (f_Level, f_Pitch);

		Tracks [i]->pmi = this;
		Tracks [i]->f_Waveform = f_Waveform;
		Tracks [i]->f_Fractal  = f_Fractal;

		Tracks [i]->Reset();
	}

  Set_Master_Info (pMasterInfo);

  if (pi != NULL)
  {
    word l_Block_ID;

    pi->Read (l_Block_ID);

    while (l_Block_ID != 0x0000)
    {
      switch (l_Block_ID)
      {
        case 0x0101 :
        {
          pi->Read (f_Listen_Enabled);
          pi->Read (f_Listen_Channel);

          break;
        }
        case 0x0201 :
        {
          Read_Controls (pi);

          break;
        }
      }

      pi->Read (l_Block_ID);
    }
  }

  if (f_Listen_Enabled)
  {
    NR_Start_Listening (f_Listen_Channel, this);
  }
}

void mi::Save (CMachineDataOutput * const po)
{
  po->Write ((word) 0x0101);
  po->Write (f_Listen_Enabled);
  po->Write (f_Listen_Channel);

  po->Write ((word) 0x0201);
  Write_Controls (po);

  po->Write ((word) 0x0000);
}

void mi::Tick ()
{
  Handle_Tick ();
}

void mi::Tick2()
{
	if (gval.attack != paraAttack.NoValue)
	{
		f_Level->Set_Attack ((float) gval.attack);
	}
	
	if (gval.decay != paraDecay.NoValue)
	{
		f_Level->Set_Decay ((float) gval.decay);
	}
	
	if (gval.sustain != paraSustain.NoValue)
	{
		f_Level->Set_Sustain (((float) gval.sustain) / ((float) 0x8000));
	}
	
	if (gval.release != paraRelease.NoValue)
	{
		f_Level->Set_Release ((float) gval.release);
	}
	
	f_Pitch->Tick (gval.bend);
	f_Fractal->Set_Fractal_Pars (gval.effect_high, gval.effect_low, gval.depth);

	if (gval.wform != paraWForm.NoValue)
	{
      f_Waveform->Set_Waveform (gval.wform);
	}

	for (int c = 0; c < numTracks; c++)
	{
		Tracks[c]->Tick(tval[c]);
	}
}

void mi::Tick2 (int p_Channel, c_Broadcast_Params *p_Command)
{
  //  Code to handle received broadcast commands
}

bool mi::Work (float *psamples, int numsamples, int const mode)
{
  return Handle_Work (psamples, numsamples, mode);
}

bool mi::Work2 (float *psamples, int numsamples, int const mode, int offset)
{
//  if ((mode & WM_WRITE) == 0)
//  {
//    return false;
//  }

	bool gotsomething = false;
	bool l_Temp;

	float *paux = pCB->GetAuxBuffer();

	for (int c = 0; c < numTracks; c++)
	{
		if (gotsomething)
		{
			l_Temp = Tracks[c]->Generate(paux, numsamples);

		    gotsomething |= l_Temp;

			if (l_Temp)
			{
				DSP_Add(psamples, paux, numsamples);
			}
		}
		else
		{
			gotsomething |= Tracks[c]->Generate(psamples, numsamples);
		}
	}

	return gotsomething;
}

void mi::Transpose_Notes  (int p_Semitones)
{
  int l_Note;

  for (int i = 0; i < MAX_TRACKS; i++)
  {
    if (   (tval [i].noteinit != NOTE_NO )
        && (tval [i].noteinit != NOTE_OFF))
    {
      l_Note  = ((tval [i].noteinit >> 4) * 12) + (tval [i].noteinit & 0x0F);
      l_Note += p_Semitones;
      
      if ((l_Note >= NOTE_MIN) && (l_Note <= NOTE_MAX))
      {
        tval [i].noteinit = ((l_Note / 12) << 4) + (l_Note % 12);
      }
      else
      {
        tval [i].noteinit = NOTE_OFF;
      }
    }

    if (   (tval [i].note != NOTE_NO )
        && (tval [i].note != NOTE_OFF))
    {
      l_Note  = ((tval [i].note >> 4) * 12) + (tval [i].note & 0x0F);
      l_Note += p_Semitones;
      
      if ((l_Note >= NOTE_MIN) && (l_Note <= NOTE_MAX))
      {
        tval [i].note = ((l_Note / 12) << 4) + (l_Note % 12);
      }
      else
      {
        tval [i].noteinit = NOTE_OFF;
        tval [i].note     = NOTE_NO;
      }
    }
  }
}

void mi::All_Notes_Off    (void)
{
  for (int i = 0; i < MAX_TRACKS; i++)
  {
    tval [i].noteinit = NOTE_OFF;
  }
}

void mi::All_Notes_Ignore (void)
{
  for (int i = 0; i < MAX_TRACKS; i++)
  {
    if (   (tval [i].noteinit != NOTE_NO )
        && (tval [i].noteinit != NOTE_OFF))
    {
      tval [i].noteinit = NOTE_NO;
    }

    tval [i].note = NOTE_NO;
  }
}

void mi::Set_Control (int p_ID, int p_Value)
{
  switch (p_ID)
  {
    case 0 :  //  Volume
    {
      for (int i = 0; i < MAX_TRACKS; i++)
      {
        int k = p_Value >> 8;

        if (k > 0xFE)  {  k = 0xFE;  }

        tval [i].volume = k;
      }

      break;
    }
    case 1 :  //  Attack
    {
      gval.attack = p_Value;

      break;
    }
    case 2 :  //  Decay
    {
      gval.decay = p_Value;

      break;
    }
    case 3 :  //  Sustain
    {
      gval.sustain = p_Value;

      break;
    }
    case 4 :  //  Release
    {
      gval.release = p_Value;

      break;
    }
    case 5 :  //  Bend Rate
    {
      gval.bend = p_Value;

      break;
    }
    case 6 :  //  Fractal Effect Low
    {
      gval.effect_low = p_Value;

      break;
    }
    case 7 :  //  Fractal Effect High
    {
      gval.effect_high = p_Value;

      break;
    }
  }
}

void CTrack::Reset()
{
	//  Should probably reset the tracks envelope here
}

void CTrack::Tick(tvals const &tv)
{
	if (tv.volume != paraVolume.NoValue)
	{
		f_Level->Set_Volume (((float) tv.volume) / ((float) 0x80));
	}

	if (tv.noteinit == NOTE_OFF)
	{
		f_Level->Stop_Note ();
	}
	else if (tv.noteinit != NOTE_NO)
	{
		f_Level->Start_Note ();
	}

	f_Pitch->Tick (tv.note,     tv.noteinit);
}

void CTrack::Stop()
{
	f_Level->Stop ();
	f_Pitch->Stop ();
}

bool CTrack::Generate(float *psamples, int numsamples)
{
	//  Derive envelope

	if (f_Level->Is_Static ())
	{
		if (f_Level->Current_Level () == 0.0)
		{
			return false;
		}
	}

	//  A constant pitch optimisation of the following 3 calls could be a good idea, though
	//  it would require two separate implementations of each waveforms inner generation
	//  loop.
	//
	//  Both pitch and amplitude envelopes might benefit from more frequent static
	//  checks, allowing an envelope to become static in the middle of a block. However,
	//  tuning of the minimum amplitude might be better.
    
	f_Pitch->Overwrite      (psamples, numsamples);
	f_Frq_To_Phase->Process (psamples, numsamples);
	f_Waveform->Process     (psamples, numsamples);

	//  Apply envelope and Fractal, and normalise amplitude

	if (f_Level->Is_Static ())
	{
		float l_Level = f_Level->Current_Level ();

		f_Fractal->Process_Static (psamples, l_Level, numsamples);

		//  Finally, apply amplitude scaling and envelope

		//  Are these locals needed?

		int	   c  = numsamples;
		float *p  = psamples;

		DSP_Amp (psamples, numsamples, l_Level * 32768.0);
	}
	else
	{
		f_Level->Overwrite (f_Buffer, numsamples);

		if (f_Fractal->Is_Static ())
		{
			//  If the fractal is static the modulator is irrelevant, so it may as well be zero
			f_Fractal->Process_Static (psamples, 0.0, numsamples);
		}
		else
		{
			f_Fractal->Process (psamples, f_Buffer, numsamples);
		}

		//  Finally, apply amplitude scaling and envelope

		NR_DSP_Multiply (psamples, f_Buffer, (float) 32768.0, numsamples);
	}

	return true;
}

#define NUM_CONTROL_CONFIGS 8

c_Control_Config g_Control_Configs [NUM_CONTROL_CONFIGS] =
{
  {0, "Volume"             },
  {1, "Attack"             },
  {2, "Sustain"            },
  {3, "Release"            },
  {4, "Decay"              },
  {5, "Bend Rate"          },
  {6, "Fractal Effect Low" },
  {7, "Fractal Effect High"}
};

void mi::Command(int const i)
{
  switch (i)
  {
    case 0 :  //  Edit Ninereeds NRS02
    {
      NR_Dialog_Enable_Channel (this, &f_Listen_Enabled, &f_Listen_Channel,
                                NUM_CONTROL_CONFIGS, g_Control_Configs     );
    }

    break;
  }
}
